/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

const float pi = 3.14159;

mat3 xrot(float t)
{
    return mat3(1.0, 0.0, 0.0,
                0.0, cos(t), -sin(t),
                0.0, sin(t), cos(t));
}

mat3 yrot(float t)
{
    return mat3(cos(t), 0.0, -sin(t),
                0.0, 1.0, 0.0,
                sin(t), 0.0, cos(t));
}

mat3 zrot(float t)
{
    return mat3(cos(t), -sin(t), 0.0,
                sin(t), cos(t), 0.0,
                0.0, 0.0, 1.0);
}

float map(vec3 p)
{
    vec3 q = fract(p) * 2.0 - 1.0;
    vec3 l = fract(p/2.0)*2.0 * 2.0 - 1.0;
    vec3 m = fract((p+vec3(0.0,0.0,1.0))/2.0)*2.0 * 2.0 - 1.0;
    vec3 u = fract(p+vec3(0.0,0.0,0.5)) * 2.0 - 1.0;
    float ts = 0.5;
    float d = max(ts-length(q.xy), ts*1.8-length(l.xz));
    d = max(d, ts-length(u.yz));
    d = min(d, p.y-0.35);

    return d;
}

vec3 normal(vec3 p)
{
    vec3 o = vec3(0.01, 0.0, 0.0);
    return normalize(vec3(map(p+o.xyy) - map(p-o.xyy),
                          map(p+o.yxy) - map(p-o.yxy),
                          map(p+o.yyx) - map(p-o.yyx)));
}

float trace(vec3 o, vec3 r)
{
    float t = 0.0;
    for (int i = 0; i < 32; ++i) {
        vec3 p = o + r * t;
        float d = map(p);
        t += d * 0.5;
    }
    return t;
}

vec3 texture(vec3 p)
{
    vec3 ta = texture2D(iChannel0, vec2(p.y,p.z)).xyz;
    vec3 tb = texture2D(iChannel0, vec2(p.x,p.z)).xyz;
    vec3 tc = texture2D(iChannel0, vec2(p.x,p.y)).xyz;
    return (ta + tb + tc) / 3.0;
}

vec3 quad(vec3 a, vec3 b, vec3 c, float t)
{
    vec3 u = mix(a, b, t);
    vec3 v = mix(b, c, t);

    return mix(u, v, t);
}

vec3 position(float b, float o)
{

    // Changed from a set of ifs to a switch when trying to
    // track down a bug. It made no difference but I stayed
    // with the switch anyway. TheEmu.

    int idx = int ( floor ( mod(b,5.0) ) );
    float t = fract(b) + o;

    switch ( idx )
     { 
       case 0 : return quad ( vec3(0.5, 0.5,-0.5),
                              vec3(0.5, 0.5, 0.5),
                              vec3(0.5, 1.5, 0.6), t );

       case 1 : return quad ( vec3(0.1, 0.5, 0.5),
                              vec3(0.5, 0.5, 0.5),
                              vec3(0.5, 0.6, 1.0), t );

       case 2 : return mix  ( vec3(0.3, 0.8, 0.6),
                              vec3(0.4, 1.3, 0.5), t );

       case 3 : return quad ( vec3(0.5, 0.5, 0.5),
                              vec3(0.1, 0.5, 0.5),
                              vec3(0.5, 0.5, 1.0), t );

       case 4 : return mix  ( vec3( 0.5, 0.5, 0.0),
                              vec3(-0.5, 0.5, 0.0), t );

     }

}

mat3 orientation(float time)
{
    // For some currently unknown reason when I run this shader
    // using my laptop's Intel Integrated Graphics Processor it
    // does not work if a and b are just the values returned by 
    // position - so I add a small offset to one of them.  This
    // hack was not needed before I upgraded to Windows 10 with
    // the change probably being related to upgraded drivers.

    vec3 a = position(time, 0.00) + 1.0e-6;
    vec3 b = position(time, 0.01);
    mat3 m;
    m[2] = normalize(b - a);
    m[1] = vec3(0.0, 1.0, 0.0);
    m[0] = normalize(cross(m[2], m[1]));
    m[2] = normalize(cross(m[1], m[0]));
    return m;
}

vec3 shade(vec3 p, vec3 r)
{
    vec3 o = p;
    vec4 fc = vec4(0.0);
    vec4 sc = vec4(0.0);
    vec3 sn = vec3(0.0);
    for (int i = 0; i < 2; ++i) {
        float t = trace(p + sn * 0.01, r);
        p = p + r * t;
        sn = normal(p);
        vec3 q = fract(p/2.0+vec3(0.0,0.0,0.5))*2.0*2.0 - 1.0;
        vec3 s = fract(p/2.0)*2.0 * 2.0 - 1.0;
        vec3 v = fract(p+0.5) * 2.0 - 1.0;
        vec4 lc = vec4(0.0);

        float prod = max(dot(r,-sn), 0.0);
        float fd = map(p);
        float fog = 1.0 / (1.0 + t * t * 0.1 + fd * 10.0);
        float pfog = prod * fog;

        if (abs(length(s.xz)-0.7) < 0.03) {
            lc = pfog * vec4(1.0, 1.0, 0.0, 0.0);
        } else if (sn.y > 0.999) {
            lc = pfog * vec4(1.0, 1.0, 1.0, 0.25);
        } else if (abs(v.x) < 0.05) {
            lc = pfog * vec4(1.0, 1.0, 0.0, 0.0);
        } else if (abs(q.z) < 0.05) {
            lc = pfog * vec4(1.0, 1.0, 0.0, 0.0);
        } else if (abs(p.y-1.0) < 0.02) {
            lc = pfog * vec4(1.0, 1.0, 0.0, 0.0);
        } else {
         lc = pfog * vec4(0.5-0.5*sin(1.0*(texture(p*yrot(pi*0.25)).x*2.0-1.0)));
            lc.w = 0.0;
        }
        r = reflect(r, -sn);
        fc = sc;
        sc = lc;
    }
   return mix(fc.xyz, sc.xyz, vec3(fc.a));
}

void main ( void )
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    uv = uv * 2.0 - 1.0;
    uv.x *= iResolution.x / iResolution.y;

    vec3 r = normalize(vec3(uv, 0.7 - dot(uv,uv) * 0.2));

    float ts = iGlobalTime * 0.125;
    r = r*orientation(ts);      // <<< PROBLEMATIC WITH INTEL GPU >>>
    vec3 o = position(ts, 0.0);

    vec3 fc = shade(o, r);

    float fade = sin(fract(ts)*pi);
    fade = 1.0 - pow(1.0 - fade, 8.0);
    fc = mix(fc, vec3(0.0, 0.0, 0.0), 1.0-fade);
    gl_FragColor = vec4(fc,1.0);
}